<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Laptop; 
use App\Models\ClassRoom;


class LaptopController extends Controller
{
    public function index(Request $request)
    {
        $query = Laptop::with(['class', 'currentAssignment.student']);

        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('asset_tag', 'like', "%{$search}%")
                  ->orWhere('barcode', 'like', "%{$search}%")
                  ->orWhere('brand', 'like', "%{$search}%")
                  ->orWhere('model', 'like', "%{$search}%")
                  ->orWhere('serial_number', 'like', "%{$search}%");
            });
        }

        if ($request->has('class_id')) {
            $query->where('class_id', $request->class_id);
        }

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        $laptops = $query->paginate(20)->withQueryString();
        $classes = ClassRoom::all();

        // Get students for searchable dropdown
        $students = \App\Models\Student::where('is_active', true)
            ->orderBy('first_name')
            ->get(['id', 'student_id', 'first_name', 'last_name', 'class_id'])
            ->map(function($student) {
                return [
                    'id' => $student->student_id,
                    'name' => $student->full_name,
                    'class_name' => $student->class ? $student->class->name : 'No Class',
                    'class_id' => $student->class_id
                ];
            });

        return view('laptops.index', compact('laptops', 'classes', 'students'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'asset_tag' => 'nullable|unique:laptops', // Made optional since it's auto-generated
            'brand' => 'required|string',
            'model' => 'required|string',
            'serial_number' => 'required|unique:laptops',
            'class_id' => 'required|exists:classes,id',
            'assigned_student_id' => 'nullable|string|max:255',
            'assigned_student_name' => 'nullable|string|max:255',
            'purchase_date' => 'nullable|date',
            'purchase_price' => 'nullable|numeric|min:0'
        ]);

        $laptop = Laptop::create($validated);

        return redirect()->route('laptops.index')
            ->with('success', 'Laptop added successfully! Asset Tag: ' . $laptop->asset_tag);
    }

    public function show(Laptop $laptop)
    {
        $laptop->load(['class', 'transactions.student', 'currentAssignment.student']);
        return view('laptops.show', compact('laptop'));
    }

    public function generateBarcodePdf($id)
    {
        $laptop = Laptop::with(['class', 'currentAssignment.student'])->findOrFail($id);
        $pdf = app('dompdf.wrapper');
        $pdf->loadView('laptops.barcode-pdf', compact('laptop'));

        return $pdf->download('barcode-' . $laptop->asset_tag . '.pdf');
    }

    public function bulkGenerateBarcodePdf(Request $request)
    {
        $request->validate([
            'laptop_ids' => 'required|array|min:1',
            'laptop_ids.*' => 'exists:laptops,id'
        ]);

        $laptopIds = $request->input('laptop_ids');
        $laptops = Laptop::whereIn('id', $laptopIds)
                        ->with(['class', 'currentAssignment.student'])
                        ->get();

        if ($laptops->isEmpty()) {
            return back()->withErrors(['laptops' => 'No valid laptops selected']);
        }

        $pdf = app('dompdf.wrapper');
        $pdf->loadView('laptops.bulk-barcode-pdf', compact('laptops'));

        $filename = 'bulk-barcodes-' . date('Y-m-d-H-i-s') . '.pdf';
        return $pdf->download($filename);
    }

    public function getByBarcode($barcode)
    {
        try {
            // Clean the barcode input
            $cleanBarcode = trim($barcode);
            
            $laptop = Laptop::where('barcode', $cleanBarcode)
                           ->with(['class', 'currentAssignment.student.class'])
                           ->first();

            if (!$laptop) {
                return response()->json([
                    'error' => 'Laptop not found with barcode: ' . $cleanBarcode
                ], 404);
            }

            return response()->json([
                'id' => $laptop->id,
                'asset_tag' => $laptop->asset_tag,
                'barcode' => $laptop->barcode,
                'brand' => $laptop->brand,
                'model' => $laptop->model,
                'status' => $laptop->status,
                'is_available' => $laptop->isAvailable(),
                'is_checked_out' => $laptop->currentAssignment !== null,
                'assigned_student_id' => $laptop->assigned_student_id,
                'assigned_student_name' => $laptop->assigned_student_name,
                'current_assignment' => $laptop->currentAssignment ? [
                    'student' => $laptop->currentAssignment->student,
                    'checked_out_at' => $laptop->currentAssignment->checked_out_at
                ] : null,
                'class' => $laptop->class
            ]);

        } catch (\Exception $e) {
            \Log::error('Error fetching laptop by barcode: ' . $e->getMessage());
            
            return response()->json([
                'error' => 'Server error while fetching laptop information'
            ], 500);
        }
    }
}
