<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\Laptop;
use App\Models\Student;
use App\Models\Transaction;
use App\Models\CurrentAssignment;
use App\Models\User;

// app/Http/Controllers/TransactionController.php
class TransactionController extends Controller
{
    public function checkOut(Request $request)
    {
        $validated = $request->validate([
            'laptop_barcode' => 'required|exists:laptops,barcode',
            'student_id' => 'required|exists:students,student_id',
            'notes' => 'nullable|string',
            'condition_before' => 'nullable|string'
        ]);

        $laptop = Laptop::where('barcode', $validated['laptop_barcode'])->first();
        $student = Student::where('student_id', $validated['student_id'])->first();

        if (!$laptop->isAvailable()) {
            return back()->withErrors(['laptop_barcode' => 'Laptop is not available']);
        }

        if ($student->currentAssignment) {
            return back()->withErrors(['student_id' => 'Student already has a laptop checked out']);
        }

        DB::transaction(function () use ($laptop, $student, $validated) {
            // Create transaction record
            Transaction::create([
                'laptop_id' => $laptop->id,
                'student_id' => $student->id,
                'user_id' => auth()->id(),
                'type' => 'check_out',
                'transaction_time' => now(),
                'notes' => $validated['notes'] ?? null,
                'condition_before' => $validated['condition_before'] ?? 'good'
            ]);

            // Create current assignment
            CurrentAssignment::create([
                'laptop_id' => $laptop->id,
                'student_id' => $student->id,
                'checked_out_at' => now(),
                'checked_out_by' => auth()->id()
            ]);

            // Update laptop status
            $laptop->update(['status' => 'checked_out']);
        });

        return redirect()->route('transactions.index')
            ->with('success', 'Laptop checked out successfully!');
    }

    public function checkIn(Request $request)
    {
        $validated = $request->validate([
            'laptop_barcode' => 'required|exists:laptops,barcode',
            'notes' => 'nullable|string',
            'condition_after' => 'nullable|string'
        ]);

        $laptop = Laptop::where('barcode', $validated['laptop_barcode'])->first();
        $assignment = $laptop->currentAssignment;

        if (!$assignment) {
            return back()->withErrors(['laptop_barcode' => 'Laptop is not currently checked out']);
        }

        DB::transaction(function () use ($laptop, $assignment, $validated) {
            // Create transaction record
            Transaction::create([
                'laptop_id' => $laptop->id,
                'student_id' => $assignment->student_id,
                'user_id' => auth()->id(),
                'type' => 'check_in',
                'transaction_time' => now(),
                'notes' => $validated['notes'] ?? null,
                'condition_after' => $validated['condition_after'] ?? 'good'
            ]);

            // Remove current assignment
            $assignment->delete();

            // Update laptop status
            $laptop->update(['status' => 'available']);
        });

        return redirect()->route('transactions.index')
            ->with('success', 'Laptop checked in successfully!');
    }

    public function checkoutForm($student_id = null)
    {
        $student = null;
        if ($student_id) {
            $student = \App\Models\Student::where('student_id', $student_id)->first();
        }

        // Get students for searchable dropdown
        $students = \App\Models\Student::where('is_active', true)
            ->orderBy('first_name')
            ->get(['id', 'student_id', 'first_name', 'last_name', 'class_id'])
            ->map(function($studentData) {
                return [
                    'id' => $studentData->student_id,
                    'name' => $studentData->full_name,
                    'class_name' => $studentData->class ? $studentData->class->name : 'No Class',
                    'class_id' => $studentData->class_id
                ];
            });

        return view('transactions.checkout', compact('student', 'students'));
    }

    public function checkinForm($laptop_barcode = null)
    {
        $laptop = null;
        if ($laptop_barcode) {
            $laptop = \App\Models\Laptop::where('barcode', $laptop_barcode)->first();
        }

        return view('transactions.checkin', compact('laptop'));
    }

    public function index(Request $request)
    {
        $query = Transaction::with(['laptop', 'student', 'user']);

        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->whereHas('laptop', function($subQ) use ($search) {
                    $subQ->where('asset_tag', 'like', "%{$search}%")
                         ->orWhere('barcode', 'like', "%{$search}%");
                })->orWhereHas('student', function($subQ) use ($search) {
                    $subQ->where('first_name', 'like', "%{$search}%")
                         ->orWhere('last_name', 'like', "%{$search}%")
                         ->orWhere('student_id', 'like', "%{$search}%");
                });
            });
        }

        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        if ($request->has('date_from')) {
            $query->whereDate('transaction_time', '>=', $request->date_from);
        }

        if ($request->has('date_to')) {
            $query->whereDate('transaction_time', '<=', $request->date_to);
        }

        $transactions = $query->orderBy('transaction_time', 'desc')->paginate(20)->withQueryString();

        // Calculate stats
        $stats = [
            'total_transactions' => Transaction::count(),
            'checkouts_today' => Transaction::where('type', 'check_out')
                ->whereDate('transaction_time', today())->count(),
            'checkins_today' => Transaction::where('type', 'check_in')
                ->whereDate('transaction_time', today())->count(),
            'currently_checked_out' => \App\Models\CurrentAssignment::count(),
        ];

        return view('transactions.index', compact('transactions', 'stats'));
    }

    public function show(Transaction $transaction)
    {
        $transaction->load(['laptop', 'student', 'user']);
        return view('transactions.show', compact('transaction'));
    }

    public function scanProcess(Request $request)
    {
        $barcode = $request->input('barcode');
        $laptop = Laptop::where('barcode', $barcode)->first();

        if (!$laptop) {
            return response()->json(['error' => 'Laptop not found'], 404);
        }

        $isCheckedOut = $laptop->currentAssignment !== null;

        return response()->json([
            'laptop' => $laptop->load(['class', 'currentAssignment.student']),
            'is_checked_out' => $isCheckedOut,
            'current_student' => $isCheckedOut ? $laptop->currentAssignment->student : null
        ]);
    }
}
