<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Laptop extends Model
{
    protected $fillable = [
        'asset_tag', 'barcode', 'brand', 'model', 'serial_number',
        'status', 'class_id', 'assigned_student_id', 'assigned_student_name',
        'notes', 'purchase_date', 'purchase_price'
    ];

    protected $casts = [
        'purchase_date' => 'date',
        'purchase_price' => 'decimal:2'
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($laptop) {
            // Auto-generate asset tag if not provided
            if (empty($laptop->asset_tag)) {
                $laptop->asset_tag = $laptop->generateAssetTag();
            }

            // Auto-generate barcode if not provided
            if (empty($laptop->barcode)) {
                $laptop->barcode = $laptop->generateBarcode();
            }
        });
    }

    public function class()
    {
        return $this->belongsTo(ClassRoom::class, 'class_id');
    }

    public function transactions()
    {
        return $this->hasMany(Transaction::class);
    }

    public function currentAssignment()
    {
        return $this->hasOne(CurrentAssignment::class);
    }

    public function isAvailable()
    {
        return $this->status === 'available' && !$this->currentAssignment;
    }

    public function generateAssetTag()
    {
        $year = date('Y');
        $count = self::whereYear('created_at', $year)->count() + 1;
        return 'LT-' . $year . '-' . str_pad($count, 3, '0', STR_PAD_LEFT);
    }

    public function generateBarcode()
    {
        if (!$this->barcode) {
            do {
                $barcode = 'LT' . str_pad(mt_rand(1, 999999), 6, '0', STR_PAD_LEFT);
            } while (self::where('barcode', $barcode)->exists());
            return $barcode;
        }
        return $this->barcode;
    }
}
