{{-- resources/views/transactions/checkin.blade.php --}}
@extends('layouts.base')

@section('content')
<div class="min-h-screen bg-gradient-to-br from-green-50 to-emerald-100 py-4 px-4 sm:px-6 lg:px-8">
    <div class="max-w-4xl mx-auto">
        <!-- Header Section -->
        <div class="text-center mb-6 sm:mb-8">
            <h2 class="text-2xl sm:text-3xl lg:text-4xl font-bold text-gray-900 mb-2">Check In Laptop</h2>
            <p class="text-sm sm:text-base text-gray-600">Scan barcode and return a laptop to inventory</p>
        </div>
        
        <div class="bg-white overflow-hidden shadow-xl rounded-xl sm:rounded-2xl">
            <div class="p-4 sm:p-6 lg:p-8">
            
            @if($errors->any())
                <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded mb-4">
                    <ul class="list-disc pl-5">
                        @foreach($errors->all() as $error)
                            <li>{{ $error }}</li>
                        @endforeach
                    </ul>
                </div>
            @endif

            <form action="{{ route('transactions.checkin.store') }}" method="POST">
                @csrf
                
                <!-- Enhanced Barcode Input Section -->
                <div class="mb-6">
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Laptop Barcode
                    </label>
                    
                    <!-- Scanner Type Indicator -->
                    <div class="mb-2">
                        <span id="scanner-status" class="text-sm text-gray-500">
                            Ready for handheld scanner or use camera below
                        </span>
                    </div>
                    
                    <div class="flex flex-col sm:flex-row gap-2 sm:gap-3">
                        <input 
                            type="text" 
                            id="laptop-barcode-input"
                            name="laptop_barcode"
                            value="{{ old('laptop_barcode', $laptop ? $laptop->barcode : '') }}"
                            placeholder="Scan barcode or enter manually" 
                            required
                            autocomplete="off"
                            class="flex-1 px-3 py-2 sm:py-3 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 font-mono text-sm sm:text-base">
                        
                        <!-- Scanner Buttons -->
                        <div class="flex gap-2">
                            <!-- Handheld Scanner Button -->
                            <button 
                                type="button" 
                                id="handheld-scan-btn"
                                class="bg-green-600 hover:bg-green-700 text-white px-3 sm:px-4 py-2 sm:py-3 rounded-md flex items-center justify-center space-x-2 text-sm sm:text-base whitespace-nowrap min-w-[100px]"
                                title="Click to focus for handheld scanner">
                                <svg class="w-4 h-4 sm:w-5 sm:h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v1m6 11h-1m-9 0H4m9-4v3m-3-3h6M12 12V8"></path>
                                </svg>
                                <span>🔌 Scanner</span>
                            </button>
                            
                            <!-- Camera Scanner Button -->
                            <button 
                                type="button" 
                                id="camera-scan-btn"
                                class="bg-blue-600 hover:bg-blue-700 text-white px-3 sm:px-4 py-2 sm:py-3 rounded-md flex items-center justify-center space-x-2 text-sm sm:text-base whitespace-nowrap min-w-[100px]"
                                title="Click to start camera scanner">
                                <svg class="w-4 h-4 sm:w-5 sm:h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 9a2 2 0 012-2h.93a2 2 0 001.664-.89l.812-1.22A2 2 0 0110.07 4h3.86a2 2 0 011.664.89l.812 1.22A2 2 0 0018.07 7H19a2 2 0 012 2v9a2 2 0 01-2 2H5a2 2 0 01-2-2V9z"></path>
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 13a3 3 0 11-6 0 3 3 0 016 0z"></path>
                                </svg>
                                <span>📷 Camera</span>
                            </button>
                        </div>
                    </div>

                    <!-- Scanner Instructions -->
                    <div class="mt-3 text-xs sm:text-sm text-gray-600">
                        <div class="flex flex-col sm:flex-row sm:items-center space-y-2 sm:space-y-0 sm:space-x-6">
                            <!-- Auto-focus toggle -->
                            <div class="flex items-center">
                                <input type="checkbox" id="auto-focus-toggle" class="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded">
                                <label for="auto-focus-toggle" class="ml-2 text-gray-700">Auto-focus for scanning</label>
                            </div>
                            <div class="flex items-center">
                                <div class="w-2 h-2 bg-green-500 rounded-full mr-2 flex-shrink-0"></div>
                                <span class="text-xs sm:text-sm"><strong>USB Scanner:</strong> Click green button, then scan</span>
                            </div>
                            <div class="flex items-center">
                                <div class="w-2 h-2 bg-blue-500 rounded-full mr-2 flex-shrink-0"></div>
                                <span class="text-xs sm:text-sm"><strong>Camera:</strong> Click blue button to scan</span>
                            </div>
                            <div class="flex items-center">
                                <div class="w-2 h-2 bg-gray-500 rounded-full mr-2 flex-shrink-0"></div>
                                <span class="text-xs sm:text-sm"><strong>Manual:</strong> Type barcode in field</span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Current Assignment Info -->
                <div id="assignment-info" class="mb-6 hidden">
                    <div class="p-4 bg-yellow-50 border border-yellow-200 rounded-md">
                        <h3 class="text-lg font-medium text-yellow-800 mb-2">Current Assignment</h3>
                        <div id="assignment-details"></div>
                    </div>
                </div>

                <!-- Condition Assessment -->
                <div class="mb-6">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Condition After Return</label>
                    <select name="condition_after" class="w-full border-gray-300 rounded-md shadow-sm">
                        <option value="excellent">Excellent</option>
                        <option value="good" selected>Good</option>
                        <option value="fair">Fair</option>
                        <option value="poor">Poor</option>
                        <option value="damaged">Damaged</option>
                    </select>
                </div>

                <!-- Damage Assessment -->
                <div id="damage-section" class="mb-6 hidden">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Damage Description</label>
                    <textarea name="damage_description" rows="3" 
                              placeholder="Describe any damage found..."
                              class="w-full border-gray-300 rounded-md shadow-sm"></textarea>
                </div>

                <!-- Notes -->
                <div class="mb-6">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Notes (Optional)</label>
                    <textarea name="notes" rows="3" 
                              placeholder="Any additional notes about the check-in..."
                              class="w-full border-gray-300 rounded-md shadow-sm">{{ old('notes') }}</textarea>
                </div>

                <!-- Submit Buttons -->
                <div class="flex flex-col sm:flex-row justify-end gap-3 sm:gap-4 pt-6 border-t border-gray-200">
                    <a href="{{ route('dashboard') }}" 
                       class="w-full sm:w-auto bg-gray-300 hover:bg-gray-400 text-gray-700 px-6 py-3 rounded-md text-center font-medium transition-colors duration-200">
                        Cancel
                    </a>
                    <button type="submit" 
                            class="w-full sm:w-auto bg-green-600 hover:bg-green-700 text-white px-6 py-3 rounded-md font-medium transition-colors duration-200">
                        Check In Laptop
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Barcode Scanner Modal -->
<div id="barcodeModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden z-50">
    <div class="flex items-center justify-center min-h-screen p-4">
        <div class="bg-white rounded-lg w-full max-w-md mx-4 sm:mx-auto p-4 sm:p-6">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-lg font-semibold">Scan Laptop Barcode</h3>
                <button id="close-camera-btn" class="text-gray-400 hover:text-gray-600 p-1">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </button>
            </div>
            <div id="qr-reader" class="w-full"></div>
            <div class="mt-4 text-xs sm:text-sm text-gray-600 text-center">
                Position the barcode within the camera view
            </div>
        </div>
    </div>
</div>

@vite('resources/js/barcode-scanner.js')
<script>
// Initialize the dual barcode scanner for checkin
document.addEventListener('DOMContentLoaded', function() {
    console.log('Initializing check-in barcode scanner...');
    
    const scanner = new DualBarcodeScanner({
        inputField: document.getElementById('laptop-barcode-input'),
        onResult: function(barcode, source) {
            console.log(`Barcode received from ${source}: ${barcode}`);
            // Update status indicator
            const statusElement = document.getElementById('scanner-status');
            if (statusElement) {
                statusElement.textContent = `Scanned from ${source}: ${barcode}`;
                statusElement.className = 'text-sm text-green-600';
            }
            
            // Show toast notification
            showToast(`Barcode scanned via ${source}`, 'success');
            
            // Validate the laptop
            validateLaptop(barcode);
        },
        onError: function(error) {
            console.error('Scanner error:', error);
            showToast('Scanner error: ' + error, 'error');
        }
    });

    // Handheld scanner button - focus input and provide feedback
    const handheldBtn = document.getElementById('handheld-scan-btn');
    if (handheldBtn) {
        handheldBtn.addEventListener('click', function() {
            const input = document.getElementById('laptop-barcode-input');
            input.focus();
            input.style.borderColor = '#10B981'; // Green border
            input.style.borderWidth = '2px';
            
            // Show temporary instruction
            showToast('Input field focused. Scan with your USB scanner now!', 'info');
            
            // Reset border after 3 seconds
            setTimeout(() => {
                input.style.borderColor = '';
                input.style.borderWidth = '';
            }, 3000);
        });
    }

    // Auto-focus toggle functionality
    const autoFocusToggle = document.getElementById('auto-focus-toggle');
    let autoFocusEnabled = false;
    
    if (autoFocusToggle) {
        autoFocusToggle.addEventListener('change', function() {
            autoFocusEnabled = this.checked;
            if (autoFocusEnabled) {
                scanner.startAutoFocus();
                showToast('Auto-focus enabled for continuous scanning', 'info');
            } else {
                scanner.stopAutoFocus();
                showToast('Auto-focus disabled - you can now edit other fields', 'info');
            }
        });
    }

    // Close camera scanner button
    const closeCameraBtn = document.getElementById('close-camera-btn');
    if (closeCameraBtn) {
        closeCameraBtn.addEventListener('click', function() {
            scanner.closeCameraScanner();
        });
    }
    
    // Check browser support on load and hide camera button if not supported
    const cameraBtn = document.getElementById('camera-scan-btn');
    if (cameraBtn) {
        if (!DualBarcodeScanner.checkBrowserSupport()) {
            cameraBtn.style.display = 'none';
            console.log('Camera not supported, hiding camera button');
        } else {
            console.log('Camera support detected');
        }
    }
    
    console.log('Check-in scanner initialization complete');
});

// Show/hide damage section based on condition
document.querySelector('select[name="condition_after"]').addEventListener('change', function() {
    const damageSection = document.getElementById('damage-section');
    if (this.value === 'damaged' || this.value === 'poor') {
        damageSection.classList.remove('hidden');
    } else {
        damageSection.classList.add('hidden');
    }
});

function validateLaptop(barcode) {
    fetch(`/api/laptops/barcode/${barcode}`, {
        method: 'GET',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            'X-Requested-With': 'XMLHttpRequest'
        },
        credentials: 'same-origin'
    })
        .then(response => response.json())
        .then(data => {
            if (data.error) {
                showError('laptop_barcode', data.error);
                hideAssignmentInfo();
            } else if (!data.is_checked_out) {
                showError('laptop_barcode', 'This laptop is not currently checked out.');
                hideAssignmentInfo();
            } else {
                clearError('laptop_barcode');
                showAssignmentInfo(data);
            }
        })
        .catch((error) => {
            console.error('Error validating laptop barcode:', error);
            console.error('Error details:', error.message);
            showError('laptop_barcode', 'Could not validate laptop barcode. Check console for details.');
            hideAssignmentInfo();
        });
}

function showAssignmentInfo(data) {
    const assignmentInfo = document.getElementById('assignment-info');
    const assignmentDetails = document.getElementById('assignment-details');
    
    const checkoutDate = new Date(data.current_assignment.checked_out_at);
    const timeAgo = getTimeAgo(checkoutDate);
    
    assignmentDetails.innerHTML = `
        <div class="grid grid-cols-1 sm:grid-cols-2 gap-3 sm:gap-4">
            <div class="text-sm">
                <strong class="text-gray-700">Student:</strong> 
                <span class="block sm:inline text-gray-900">${data.current_assignment.student.full_name}</span>
            </div>
            <div class="text-sm">
                <strong class="text-gray-700">Student ID:</strong> 
                <span class="block sm:inline text-gray-900">${data.current_assignment.student.student_id}</span>
            </div>
            <div class="text-sm">
                <strong class="text-gray-700">Class:</strong> 
                <span class="block sm:inline text-gray-900">${data.current_assignment.student.class ? data.current_assignment.student.class.name : 'No Class'}</span>
            </div>
            <div class="text-sm">
                <strong class="text-gray-700">Laptop:</strong> 
                <span class="block sm:inline text-gray-900">${data.asset_tag} (${data.brand} ${data.model})</span>
            </div>
        </div>
        <div class="mt-3 pt-3 border-t border-yellow-200">
            <div class="grid grid-cols-1 sm:grid-cols-2 gap-2 text-sm">
                <div>
                    <strong class="text-gray-700">Checked Out:</strong> 
                    <span class="block sm:inline text-gray-900">${checkoutDate.toLocaleDateString()} at ${checkoutDate.toLocaleTimeString()}</span>
                </div>
                <div>
                    <strong class="text-gray-700">Duration:</strong> 
                    <span class="text-yellow-700 font-medium">${timeAgo}</span>
                </div>
            </div>
        </div>
    `;
    
    // Update container styling
    const container = assignmentInfo.querySelector('.p-4');
    container.className = 'p-3 sm:p-4 bg-yellow-50 border border-yellow-200 rounded-md';
    
    assignmentInfo.classList.remove('hidden');
}

function hideAssignmentInfo() {
    document.getElementById('assignment-info').classList.add('hidden');
}

function getTimeAgo(date) {
    const now = new Date();
    const diffInMs = now - date;
    const diffInDays = Math.floor(diffInMs / (1000 * 60 * 60 * 24));
    const diffInHours = Math.floor(diffInMs / (1000 * 60 * 60));
    const diffInMinutes = Math.floor(diffInMs / (1000 * 60));
    
    if (diffInDays > 0) {
        return `${diffInDays} day${diffInDays > 1 ? 's' : ''} ago`;
    } else if (diffInHours > 0) {
        return `${diffInHours} hour${diffInHours > 1 ? 's' : ''} ago`;
    } else {
        return `${diffInMinutes} minute${diffInMinutes > 1 ? 's' : ''} ago`;
    }
}

function showError(fieldName, message) {
    const field = document.querySelector(`input[name="${fieldName}"]`);
    let errorDiv = field.parentNode.querySelector('.error-message');
    
    if (!errorDiv) {
        errorDiv = document.createElement('div');
        errorDiv.className = 'error-message mt-1 text-sm text-red-600';
        field.parentNode.appendChild(errorDiv);
    }
    
    errorDiv.textContent = message;
    field.classList.add('border-red-300');
}

function clearError(fieldName) {
    const field = document.querySelector(`input[name="${fieldName}"]`);
    const errorDiv = field.parentNode.querySelector('.error-message');
    
    if (errorDiv) {
        errorDiv.remove();
    }
    
    field.classList.remove('border-red-300');
}
</script>
@endsection