{{-- resources/views/transactions/checkout.blade.php --}}
@extends('layouts.base')

@section('content')
<div class="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 py-4 px-4 sm:px-6 lg:px-8">
    <div class="max-w-4xl mx-auto">
        <!-- Header Section -->
        <div class="text-center mb-6 sm:mb-8">
            <h2 class="text-2xl sm:text-3xl lg:text-4xl font-bold text-gray-900 mb-2">Check Out Laptop</h2>
            <p class="text-sm sm:text-base text-gray-600">Scan barcode or search to assign a laptop to a student</p>
        </div>
        
        <div class="bg-white overflow-hidden shadow-xl rounded-xl sm:rounded-2xl">
            <div class="p-4 sm:p-6 lg:p-8">
            
            @if($errors->any())
                <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded mb-4">
                    <ul class="list-disc pl-5">
                        @foreach($errors->all() as $error)
                            <li>{{ $error }}</li>
                        @endforeach
                    </ul>
                </div>
            @endif

            <form action="{{ route('transactions.checkout.store') }}" method="POST">
                @csrf
                
                <!-- Enhanced Barcode Input Section -->
                <div class="mb-6">
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Laptop Barcode
                    </label>
                    
                    <!-- Scanner Type Indicator -->
                    <div class="mb-2">
                        <span id="scanner-status" class="text-sm text-gray-500">
                            Ready for handheld scanner or use camera below
                        </span>
                    </div>
                    
                    <div class="flex flex-col sm:flex-row gap-2 sm:gap-3">
                        <input 
                            type="text" 
                            id="laptop-barcode-input"
                            name="laptop_barcode"
                            value="{{ old('laptop_barcode', request('laptop_barcode')) }}"
                            placeholder="Scan barcode or enter manually" 
                            required
                            autocomplete="off"
                            class="flex-1 px-3 py-2 sm:py-3 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 font-mono text-sm sm:text-base">
                        
                        <!-- Scanner Buttons -->
                        <div class="flex gap-2">
                            <!-- Handheld Scanner Button -->
                            <button 
                                type="button" 
                                id="handheld-scan-btn"
                                class="bg-green-600 hover:bg-green-700 text-white px-3 sm:px-4 py-2 sm:py-3 rounded-md flex items-center justify-center space-x-2 text-sm sm:text-base whitespace-nowrap min-w-[100px]"
                                title="Click to focus for handheld scanner">
                                <svg class="w-4 h-4 sm:w-5 sm:h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v1m6 11h-1m-9 0H4m9-4v3m-3-3h6M12 12V8"></path>
                                </svg>
                                <span>🔌 Scanner</span>
                            </button>
                            
                            <!-- Camera Scanner Button -->
                            <button 
                                type="button" 
                                id="camera-scan-btn"
                                class="bg-blue-600 hover:bg-blue-700 text-white px-3 sm:px-4 py-2 sm:py-3 rounded-md flex items-center justify-center space-x-2 text-sm sm:text-base whitespace-nowrap min-w-[100px]"
                                title="Click to start camera scanner">
                                <svg class="w-4 h-4 sm:w-5 sm:h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 9a2 2 0 012-2h.93a2 2 0 001.664-.89l.812-1.22A2 2 0 0110.07 4h3.86a2 2 0 011.664.89l.812 1.22A2 2 0 0018.07 7H19a2 2 0 012 2v9a2 2 0 01-2 2H5a2 2 0 01-2-2V9z"></path>
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 13a3 3 0 11-6 0 3 3 0 016 0z"></path>
                                </svg>
                                <span>📷 Camera</span>
                            </button>
                        </div>
                    </div>

                    <!-- Scanner Instructions -->
                    <div class="mt-3 text-xs sm:text-sm text-gray-600">
                        <div class="flex flex-col sm:flex-row sm:items-center space-y-2 sm:space-y-0 sm:space-x-6">
                            <!-- Auto-focus toggle -->
                            <div class="flex items-center">
                                <input type="checkbox" id="auto-focus-toggle" class="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded">
                                <label for="auto-focus-toggle" class="ml-2 text-gray-700">Auto-focus for scanning</label>
                            </div>
                            <div class="flex items-center">
                                <div class="w-2 h-2 bg-green-500 rounded-full mr-2 flex-shrink-0"></div>
                                <span class="text-xs sm:text-sm"><strong>USB Scanner:</strong> Click green button, then scan</span>
                            </div>
                            <div class="flex items-center">
                                <div class="w-2 h-2 bg-blue-500 rounded-full mr-2 flex-shrink-0"></div>
                                <span class="text-xs sm:text-sm"><strong>Camera:</strong> Click blue button to scan</span>
                            </div>
                            <div class="flex items-center">
                                <div class="w-2 h-2 bg-gray-500 rounded-full mr-2 flex-shrink-0"></div>
                                <span class="text-xs sm:text-sm"><strong>Manual:</strong> Type barcode in field</span>
                            </div>
                        </div>
                    </div>
                </div>

                    <!-- Laptop Info Display -->
                    <div id="laptop-info" class="mt-2 {{ request('laptop_barcode') ? '' : 'hidden' }}">
                        <div class="p-3 bg-blue-50 border border-blue-200 rounded-md">
                            <h4 class="text-sm font-semibold text-blue-800 mb-2">Laptop Information</h4>
                            <div id="laptop-details">
                                @if(request('laptop_barcode'))
                                    <p class="text-sm text-blue-700">Barcode: {{ request('laptop_barcode') }}</p>
                                    <div id="laptop-full-info" class="hidden">
                                        <!-- This will be populated by JavaScript -->
                                    </div>
                                @endif
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Student Selection -->
                <div class="mb-6">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Search Student</label>

                    <!-- Searchable input -->
                    <div class="relative">
                        <input type="text"
                               id="checkout_student_search_input"
                               placeholder="Type student name or ID..."
                               class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                               autocomplete="off"
                               required>

                        <!-- Search results dropdown -->
                        <div id="checkout_student_search_results"
                             class="absolute z-10 w-full bg-white border border-gray-300 rounded-md shadow-lg max-h-60 overflow-y-auto hidden">
                            <div id="checkout_student_results_list">
                                <!-- Results will be populated by JavaScript -->
                            </div>
                        </div>
                    </div>

                    <!-- Selected student display -->
                    <div id="checkout_selected_student_display" class="hidden mt-2 p-3 bg-green-50 border border-green-200 rounded-md">
                        <div class="flex items-center justify-between">
                            <div>
                                <h4 class="text-sm font-semibold text-green-800 mb-2">Selected Student:</h4>
                                <div id="checkout_selected_student_info" class="text-sm text-green-700"></div>
                            </div>
                            <button type="button"
                                    id="checkout_clear_student_selection"
                                    class="text-red-600 hover:text-red-800 text-sm">
                                Clear
                            </button>
                        </div>
                    </div>

                    <!-- Hidden field for form submission -->
                    <input type="hidden" name="student_id" id="checkout_student_id">
                </div>

                <!-- Condition Assessment -->
                <div class="mb-6">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Condition Before Checkout</label>
                    <select name="condition_before" class="w-full border-gray-300 rounded-md shadow-sm">
                        <option value="excellent">Excellent</option>
                        <option value="good" selected>Good</option>
                        <option value="fair">Fair</option>
                        <option value="poor">Poor</option>
                    </select>
                </div>

                <!-- Notes -->
                <div class="mb-6">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Notes (Optional)</label>
                    <textarea name="notes" rows="3" 
                              placeholder="Any additional notes about the checkout..."
                              class="w-full border-gray-300 rounded-md shadow-sm">{{ old('notes') }}</textarea>
                </div>

                <!-- Submit Buttons -->
                <div class="flex flex-col sm:flex-row justify-end gap-3 sm:gap-4 pt-6 border-t border-gray-200">
                    <a href="{{ route('dashboard') }}" 
                       class="w-full sm:w-auto bg-gray-300 hover:bg-gray-400 text-gray-700 px-6 py-3 rounded-md text-center font-medium transition-colors duration-200">
                        Cancel
                    </a>
                    <button type="submit" 
                            class="w-full sm:w-auto bg-blue-600 hover:bg-blue-700 text-white px-6 py-3 rounded-md font-medium transition-colors duration-200">
                        Check Out Laptop
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Barcode Scanner Modal -->
<div id="barcodeModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden z-50">
    <div class="flex items-center justify-center min-h-screen p-4">
        <div class="bg-white rounded-lg w-full max-w-md mx-4 sm:mx-auto p-4 sm:p-6">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-lg font-semibold">Scan Laptop Barcode</h3>
                <button id="close-camera-btn" class="text-gray-400 hover:text-gray-600 p-1">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </button>
            </div>
            <div id="qr-reader" class="w-full"></div>
            <div class="mt-4 text-xs sm:text-sm text-gray-600 text-center">
                Position the barcode within the camera view
            </div>
        </div>
    </div>
</div>

@vite('resources/js/barcode-scanner.js')
<script>
// Initialize the dual barcode scanner
document.addEventListener('DOMContentLoaded', function() {
    console.log('Initializing barcode scanner...');
    
    const scanner = new DualBarcodeScanner({
        inputField: document.getElementById('laptop-barcode-input'),
        onResult: function(barcode, source) {
            console.log(`Barcode received from ${source}: ${barcode}`);
            // Update status indicator
            const statusElement = document.getElementById('scanner-status');
            if (statusElement) {
                statusElement.textContent = `Scanned from ${source}: ${barcode}`;
                statusElement.className = 'text-sm text-green-600';
            }
            
            // Show toast notification
            showToast(`Barcode scanned via ${source}`, 'success');
            
            // Validate the laptop
            validateLaptop(barcode);
        },
        onError: function(error) {
            console.error('Scanner error:', error);
            showToast('Scanner error: ' + error, 'error');
        }
    });

    // Handheld scanner button - focus input and provide feedback
    const handheldBtn = document.getElementById('handheld-scan-btn');
    if (handheldBtn) {
        handheldBtn.addEventListener('click', function() {
            const input = document.getElementById('laptop-barcode-input');
            input.focus();
            input.style.borderColor = '#10B981'; // Green border
            input.style.borderWidth = '2px';
            
            // Show temporary instruction
            showToast('Input field focused. Scan with your USB scanner now!', 'info');
            
            // Reset border after 3 seconds
            setTimeout(() => {
                input.style.borderColor = '';
                input.style.borderWidth = '';
            }, 3000);
        });
    }

    // Auto-focus toggle functionality
    const autoFocusToggle = document.getElementById('auto-focus-toggle');
    let autoFocusEnabled = false;
    
    if (autoFocusToggle) {
        autoFocusToggle.addEventListener('change', function() {
            autoFocusEnabled = this.checked;
            if (autoFocusEnabled) {
                scanner.startAutoFocus();
                showToast('Auto-focus enabled for continuous scanning', 'info');
            } else {
                scanner.stopAutoFocus();
                showToast('Auto-focus disabled - you can now edit other fields', 'info');
            }
        });
    }

    // Close camera scanner button
    const closeCameraBtn = document.getElementById('close-camera-btn');
    if (closeCameraBtn) {
        closeCameraBtn.addEventListener('click', function() {
            scanner.closeCameraScanner();
        });
    }
    
    // Check browser support on load and hide camera button if not supported
    const cameraBtn = document.getElementById('camera-scan-btn');
    if (cameraBtn) {
        if (!DualBarcodeScanner.checkBrowserSupport()) {
            cameraBtn.style.display = 'none';
            console.log('Camera not supported, hiding camera button');
        } else {
            console.log('Camera support detected');
        }
    }
    
    console.log('Scanner initialization complete');
});

// Student data for searchable dropdown
let checkoutAllStudents = @json($students);

function validateLaptop(barcode) {
    console.log('validateLaptop called with barcode:', barcode);
    fetch(`/api/laptops/barcode/${barcode}`, {
        method: 'GET',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            'X-Requested-With': 'XMLHttpRequest'
        },
        credentials: 'same-origin'
    })
        .then(response => {
            console.log('Fetch response status:', response.status, response.statusText);
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            return response.json();
        })
        .then(data => {
            console.log('Laptop data:', data); // Debug log
            if (data.error) {
                showError('laptop_barcode', data.error);
                hideLaptopInfo();
            } else {
                clearError('laptop_barcode');
                showLaptopInfo(data);

                // If laptop is checked out, prefill student information
                if (data.is_checked_out && data.current_assignment) {
                    console.log('Prefilling checked out student'); // Debug log
                    const student = data.current_assignment.student;
                    const searchInput = document.getElementById('checkout_student_search_input');
                    const studentIdInput = document.getElementById('checkout_student_id');
                    const selectedDisplay = document.getElementById('checkout_selected_student_display');
                    const selectedInfo = document.getElementById('checkout_selected_student_info');

                    // Update form field
                    studentIdInput.value = student.student_id;

                    // Update display
                    selectedInfo.innerHTML = `
                        <div><strong>Name:</strong> ${student.first_name} ${student.last_name}</div>
                        <div><strong>Student ID:</strong> ${student.student_id}</div>
                        <div><strong>Class:</strong> ${student.class ? student.class.name : 'No Class'}</div>
                    `;

                    // Update search input
                    searchInput.value = `${student.first_name} ${student.last_name} (${student.student_id})`;

                    // Show selected display
                    selectedDisplay.classList.remove('hidden');
                }
                // If laptop is assigned but not checked out, prefill assigned student information
                else if (!data.is_checked_out && data.assigned_student_id && data.assigned_student_name && data.assigned_student_id.trim() !== '') {
                    console.log('Prefilling assigned student:', data.assigned_student_id, data.assigned_student_name); // Debug log
                    const searchInput = document.getElementById('checkout_student_search_input');
                    const studentIdInput = document.getElementById('checkout_student_id');
                    const selectedDisplay = document.getElementById('checkout_selected_student_display');
                    const selectedInfo = document.getElementById('checkout_selected_student_info');

                    // Update form field
                    studentIdInput.value = data.assigned_student_id;

                    // Update display
                    selectedInfo.innerHTML = `
                        <div><strong>Name:</strong> ${data.assigned_student_name}</div>
                        <div><strong>Student ID:</strong> ${data.assigned_student_id}</div>
                        <div><strong>Class:</strong> Assigned Laptop</div>
                    `;

                    // Update search input
                    searchInput.value = `${data.assigned_student_name} (${data.assigned_student_id})`;

                    // Show selected display
                    selectedDisplay.classList.remove('hidden');
                } else {
                    console.log('No prefilling:');
                    console.log('  is_checked_out:', data.is_checked_out, typeof data.is_checked_out);
                    console.log('  assigned_student_id:', data.assigned_student_id, typeof data.assigned_student_id);
                    console.log('  assigned_student_name:', data.assigned_student_name, typeof data.assigned_student_name);
                    console.log('  assigned_student_id trimmed:', data.assigned_student_id ? data.assigned_student_id.trim() : 'N/A');
                    console.log('  Condition checks:');
                    console.log('    !data.is_checked_out:', !data.is_checked_out);
                    console.log('    data.assigned_student_id exists:', !!data.assigned_student_id);
                    console.log('    data.assigned_student_name exists:', !!data.assigned_student_name);
                    console.log('    assigned_student_id.trim() !== "":', data.assigned_student_id ? data.assigned_student_id.trim() !== '' : false);
                }
            }
        })
        .catch((error) => {
            console.error('Error validating laptop barcode:', error);
            console.error('Error details:', error.message);
            showError('laptop_barcode', 'Could not validate laptop barcode. Check console for details.');
            hideLaptopInfo();
        });
}

function showLaptopInfo(data) {
    const laptopInfo = document.getElementById('laptop-info');
    const laptopDetails = document.getElementById('laptop-details');

    let html = `
        <div class="grid grid-cols-1 sm:grid-cols-2 gap-3 sm:gap-4">
            <div class="text-sm">
                <strong class="text-gray-700">Asset Tag:</strong> 
                <span class="block sm:inline text-gray-900">${data.asset_tag}</span>
            </div>
            <div class="text-sm">
                <strong class="text-gray-700">Brand:</strong> 
                <span class="block sm:inline text-gray-900">${data.brand}</span>
            </div>
            <div class="text-sm">
                <strong class="text-gray-700">Model:</strong> 
                <span class="block sm:inline text-gray-900">${data.model}</span>
            </div>
            <div class="text-sm">
                <strong class="text-gray-700">Status:</strong> 
                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${getStatusColor(data.status)}">
                    ${data.status.replace('_', ' ').toUpperCase()}
                </span>
            </div>
        </div>
    `;

    if (data.class) {
        html += `<p class="text-sm text-blue-700 mt-3 sm:mt-2"><strong>Class:</strong> ${data.class.name}</p>`;
    }

    if (data.assigned_student_id || data.assigned_student_name) {
        html += '<div class="mt-3 pt-3 border-t border-blue-200">';
        html += '<p class="text-sm font-semibold text-blue-800 mb-2">Assigned Student:</p>';
        html += '<div class="grid grid-cols-1 sm:grid-cols-2 gap-2 text-sm">';
        if (data.assigned_student_id) {
            html += `<div><strong class="text-gray-700">ID:</strong> <span class="text-blue-700">${data.assigned_student_id}</span></div>`;
        }
        if (data.assigned_student_name) {
            html += `<div><strong class="text-gray-700">Name:</strong> <span class="text-blue-700">${data.assigned_student_name}</span></div>`;
        }
        html += '</div></div>';
    }

    if (data.is_checked_out && data.current_assignment) {
        const student = data.current_assignment.student;
        const checkoutDate = new Date(data.current_assignment.checked_out_at);
        html += '<div class="mt-3 pt-3 border-t border-red-200">';
        html += '<p class="text-sm font-semibold text-red-800 mb-2">⚠️ Currently Checked Out:</p>';
        html += '<div class="grid grid-cols-1 sm:grid-cols-2 gap-2 text-sm">';
        html += `<div><strong class="text-gray-700">Student:</strong> <span class="text-red-700">${student.first_name} ${student.last_name}</span></div>`;
        html += `<div><strong class="text-gray-700">Student ID:</strong> <span class="text-red-700">${student.student_id}</span></div>`;
        html += `<div class="col-span-1 sm:col-span-2"><strong class="text-gray-700">Checked Out:</strong> <span class="text-red-700">${checkoutDate.toLocaleDateString()}</span></div>`;
        html += '</div></div>';
    }

    laptopDetails.innerHTML = html;
    
    // Update container styling based on laptop status
    const container = laptopInfo.querySelector('.p-3');
    if (data.is_checked_out) {
        container.className = 'p-3 sm:p-4 bg-yellow-50 border border-yellow-200 rounded-md';
    } else if (data.status === 'available') {
        container.className = 'p-3 sm:p-4 bg-green-50 border border-green-200 rounded-md';
    } else {
        container.className = 'p-3 sm:p-4 bg-red-50 border border-red-200 rounded-md';
    }
    
    laptopInfo.classList.remove('hidden');

    // Laptop validation successful - user can now proceed to student selection
    if (data.status === 'available') {
        // Removed auto-focus to allow users to fill other fields first
    }
}

function hideLaptopInfo() {
    document.getElementById('laptop-info').classList.add('hidden');
}

function validateStudentCheckout(studentId) {
    fetch(`/api/students/search?student_id=${studentId}`, {
        method: 'GET',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            'X-Requested-With': 'XMLHttpRequest'
        },
        credentials: 'same-origin'
    })
        .then(response => response.json())
        .then(data => {
            if (data.has_laptop) {
                showError('student_id', 'This student already has a laptop checked out.');
            } else {
                clearError('student_id');
            }
        })
        .catch(() => {
            showError('student_id', 'Could not validate student.');
        });
}

// Handle searchable student selection for checkout
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('checkout_student_search_input');
    const resultsContainer = document.getElementById('checkout_student_search_results');
    const resultsList = document.getElementById('checkout_student_results_list');
    const selectedDisplay = document.getElementById('checkout_selected_student_display');
    const selectedInfo = document.getElementById('checkout_selected_student_info');
    const studentIdInput = document.getElementById('checkout_student_id');
    const clearButton = document.getElementById('checkout_clear_student_selection');

    let selectedStudent = null;

    // Search functionality
    searchInput.addEventListener('input', function() {
        const query = this.value.toLowerCase().trim();

        if (query.length < 2) {
            resultsContainer.classList.add('hidden');
            return;
        }

        const filteredStudents = checkoutAllStudents.filter(student =>
            student.name.toLowerCase().includes(query) ||
            student.id.toLowerCase().includes(query) ||
            student.class_name.toLowerCase().includes(query)
        );

        displaySearchResults(filteredStudents);
        resultsContainer.classList.remove('hidden');
    });

    // Hide results when clicking outside
    document.addEventListener('click', function(e) {
        if (!searchInput.contains(e.target) && !resultsContainer.contains(e.target)) {
            resultsContainer.classList.add('hidden');
        }
    });

    // Clear selection
    clearButton.addEventListener('click', function() {
        clearSelection();
    });

    function displaySearchResults(students) {
        if (students.length === 0) {
            resultsList.innerHTML = '<div class="p-3 text-gray-500 text-sm">No students found</div>';
            return;
        }

        const html = students.slice(0, 10).map(student => `
            <div class="student-result p-3 hover:bg-green-50 cursor-pointer border-b border-gray-100 last:border-b-0"
                 data-student-id="${student.id}"
                 data-student-name="${student.name}"
                 data-class-name="${student.class_name}">
                <div class="font-medium text-gray-900">${student.name}</div>
                <div class="text-sm text-gray-600">ID: ${student.id} • Class: ${student.class_name}</div>
            </div>
        `).join('');

        resultsList.innerHTML = html;

        // Add click handlers to results
        document.querySelectorAll('.student-result').forEach(result => {
            result.addEventListener('click', function() {
                const studentId = this.getAttribute('data-student-id');
                const studentName = this.getAttribute('data-student-name');
                const className = this.getAttribute('data-class-name');

                selectStudent(studentId, studentName, className);
            });
        });
    }

    function selectStudent(studentId, studentName, className) {
        // Update form field
        studentIdInput.value = studentId;

        // Update display
        selectedInfo.innerHTML = `
            <div><strong>Name:</strong> ${studentName}</div>
            <div><strong>Student ID:</strong> ${studentId}</div>
            <div><strong>Class:</strong> ${className}</div>
        `;

        // Update search input
        searchInput.value = `${studentName} (${studentId})`;

        // Show selected display, hide results
        selectedDisplay.classList.remove('hidden');
        resultsContainer.classList.add('hidden');

        // Validate if student already has a laptop
        validateStudentCheckout(studentId);

        selectedStudent = { id: studentId, name: studentName, className };
    }

    function clearSelection() {
        studentIdInput.value = '';
        searchInput.value = '';
        selectedDisplay.classList.add('hidden');
        resultsContainer.classList.add('hidden');
        selectedStudent = null;
    }
});

function showError(fieldName, message) {
    let field;
    if (fieldName === 'student_id') {
        field = document.getElementById('checkout_student_search_input');
    } else {
        field = document.querySelector(`[name="${fieldName}"]`);
    }

    if (!field) return;

    let errorDiv = field.parentNode.querySelector('.error-message');

    if (!errorDiv) {
        errorDiv = document.createElement('div');
        errorDiv.className = 'error-message mt-1 text-sm text-red-600';
        field.parentNode.appendChild(errorDiv);
    }

    errorDiv.textContent = message;
    field.classList.add('border-red-300');
}

function clearError(fieldName) {
    let field;
    if (fieldName === 'student_id') {
        field = document.getElementById('checkout_student_search_input');
    } else {
        field = document.querySelector(`[name="${fieldName}"]`);
    }

    if (!field) return;

    const errorDiv = field.parentNode.querySelector('.error-message');

    if (errorDiv) {
        errorDiv.remove();
    }

    field.classList.remove('border-red-300');
}
</script>
@endsection