{{-- resources/views/transactions/index.blade.php --}}
@extends('layouts.base')

@section('content')
<div class="max-w-7xl mx-auto sm:px-6 lg:px-8">
    <div class="bg-white overflow-hidden shadow-sm sm:rounded-lg">
        <div class="p-6 bg-white border-b border-gray-200">
            <!-- Header -->
            <div class="flex justify-between items-center mb-6">
                <h2 class="text-2xl font-bold text-gray-900">Transaction History</h2>
                <div class="flex space-x-4">
                    <a href="{{ route('transactions.checkout') }}" 
                       class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-md">
                        Check Out
                    </a>
                    <a href="{{ route('transactions.checkin') }}" 
                       class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded-md">
                        Check In
                    </a>
                </div>
            </div>

            <!-- Filters -->
            <form method="GET" action="{{ route('transactions.index') }}" class="mb-6">
                <div class="grid grid-cols-1 md:grid-cols-5 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Date From</label>
                        <input type="date" name="date_from" value="{{ request('date_from') }}" 
                               class="w-full border-gray-300 rounded-md shadow-sm">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Date To</label>
                        <input type="date" name="date_to" value="{{ request('date_to') }}" 
                               class="w-full border-gray-300 rounded-md shadow-sm">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Type</label>
                        <select name="type" class="w-full border-gray-300 rounded-md shadow-sm">
                            <option value="">All Types</option>
                            <option value="check_out" {{ request('type') == 'check_out' ? 'selected' : '' }}>Check Out</option>
                            <option value="check_in" {{ request('type') == 'check_in' ? 'selected' : '' }}>Check In</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Student</label>
                        <input type="text" name="student_search" value="{{ request('student_search') }}" 
                               placeholder="Student name or ID"
                               class="w-full border-gray-300 rounded-md shadow-sm">
                    </div>
                    <div class="flex items-end">
                        <button type="submit" class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 rounded-md mr-2">
                            Filter
                        </button>
                        <a href="{{ route('transactions.index') }}" class="text-gray-500 hover:text-gray-700">
                            Clear
                        </a>
                    </div>
                </div>
            </form>

            <!-- Summary Stats -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
                <div class="bg-blue-50 p-4 rounded-lg">
                    <div class="text-2xl font-bold text-blue-600">{{ $stats['total_transactions'] ?? 0 }}</div>
                    <div class="text-sm text-blue-800">Total Transactions</div>
                </div>
                <div class="bg-yellow-50 p-4 rounded-lg">
                    <div class="text-2xl font-bold text-yellow-600">{{ $stats['checkouts_today'] ?? 0 }}</div>
                    <div class="text-sm text-yellow-800">Checkouts Today</div>
                </div>
                <div class="bg-green-50 p-4 rounded-lg">
                    <div class="text-2xl font-bold text-green-600">{{ $stats['checkins_today'] ?? 0 }}</div>
                    <div class="text-sm text-green-800">Check-ins Today</div>
                </div>
                <div class="bg-purple-50 p-4 rounded-lg">
                    <div class="text-2xl font-bold text-purple-600">{{ $stats['currently_checked_out'] ?? 0 }}</div>
                    <div class="text-sm text-purple-800">Currently Out</div>
                </div>
            </div>

            <!-- Transactions Table -->
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Date/Time
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Type
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Laptop
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Student
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Staff Member
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Details
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Actions
                            </th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        @forelse($transactions as $transaction)
                            <tr class="hover:bg-gray-50">
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm font-medium text-gray-900">
                                        {{ $transaction->transaction_time->format('M d, Y') }}
                                    </div>
                                    <div class="text-sm text-gray-500">
                                        {{ $transaction->transaction_time->format('H:i') }}
                                        <span class="text-xs text-gray-400">
                                            ({{ $transaction->transaction_time->diffForHumans() }})
                                        </span>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full
                                        {{ $transaction->type === 'check_out' ? 'bg-yellow-100 text-yellow-800' : 'bg-green-100 text-green-800' }}">
                                        {{ $transaction->type === 'check_out' ? 'Check Out' : 'Check In' }}
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="flex items-center">
                                        <div>
                                            <div class="text-sm font-medium text-gray-900">
                                                {{ $transaction->laptop->asset_tag }}
                                            </div>
                                            <div class="text-sm text-gray-500">
                                                {{ $transaction->laptop->brand }} {{ $transaction->laptop->model }}
                                            </div>
                                        </div>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <div class="flex items-center">
                                        <div>
                                            <div class="text-sm font-medium text-gray-900">
                                                {{ $transaction->student->full_name }}
                                            </div>
                                            <div class="text-sm text-gray-500">
                                                {{ $transaction->student->student_id }} - {{ $transaction->student->class->name ?? 'No Class' }}
                                            </div>
                                        </div>
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                    {{ $transaction->user->name }}
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm text-gray-900">
                                        @if($transaction->condition_before)
                                            <div class="mb-1">
                                                <span class="text-xs text-gray-500">Before:</span>
                                                <span class="px-2 inline-flex text-xs leading-4 font-medium rounded-full
                                                    {{ $transaction->condition_before === 'excellent' ? 'bg-green-100 text-green-800' : '' }}
                                                    {{ $transaction->condition_before === 'good' ? 'bg-blue-100 text-blue-800' : '' }}
                                                    {{ $transaction->condition_before === 'fair' ? 'bg-yellow-100 text-yellow-800' : '' }}
                                                    {{ in_array($transaction->condition_before, ['poor', 'damaged']) ? 'bg-red-100 text-red-800' : '' }}">
                                                    {{ ucfirst($transaction->condition_before) }}
                                                </span>
                                            </div>
                                        @endif
                                        @if($transaction->condition_after)
                                            <div class="mb-1">
                                                <span class="text-xs text-gray-500">After:</span>
                                                <span class="px-2 inline-flex text-xs leading-4 font-medium rounded-full
                                                    {{ $transaction->condition_after === 'excellent' ? 'bg-green-100 text-green-800' : '' }}
                                                    {{ $transaction->condition_after === 'good' ? 'bg-blue-100 text-blue-800' : '' }}
                                                    {{ $transaction->condition_after === 'fair' ? 'bg-yellow-100 text-yellow-800' : '' }}
                                                    {{ in_array($transaction->condition_after, ['poor', 'damaged']) ? 'bg-red-100 text-red-800' : '' }}">
                                                    {{ ucfirst($transaction->condition_after) }}
                                                </span>
                                            </div>
                                        @endif
                                        @if($transaction->damage_description)
                                            <div class="text-xs text-red-600 bg-red-50 p-2 rounded mt-1">
                                                <strong>Damage:</strong> {{ $transaction->damage_description }}
                                            </div>
                                        @endif
                                        @if($transaction->notes)
                                            <div class="text-xs text-gray-600 mt-1">
                                                <strong>Notes:</strong> {{ Str::limit($transaction->notes, 50) }}
                                            </div>
                                        @endif
                                    </div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                    <div class="flex space-x-2">
                                        <button onclick="showTransactionDetails({{ $transaction->id }})" 
                                                class="text-indigo-600 hover:text-indigo-900">View</button>
                                        @if($transaction->type === 'check_out' && !$transaction->laptop->currentAssignment)
                                            <span class="text-gray-400">Returned</span>
                                        @elseif($transaction->type === 'check_out')
                                            <a href="{{ route('transactions.checkin', ['laptop_barcode' => $transaction->laptop->barcode]) }}" 
                                               class="text-red-600 hover:text-red-900">Check In</a>
                                        @endif
                                    </div>
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="7" class="px-6 py-4 text-center text-gray-500">
                                    No transactions found matching your criteria.
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            <div class="mt-6">
                {{ $transactions->appends(request()->query())->links() }}
            </div>
        </div>
    </div>
</div>

<!-- Transaction Details Modal -->
<div id="transactionModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden z-50">
    <div class="flex items-center justify-center min-h-screen p-4">
        <div class="bg-white rounded-lg max-w-2xl w-full p-6">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-lg font-semibold">Transaction Details</h3>
                <button onclick="closeTransactionModal()" class="text-gray-400 hover:text-gray-600">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </button>
            </div>
            
            <div id="transactionDetails" class="space-y-4">
                <!-- Details will be loaded here -->
            </div>
            
            <div class="mt-6 flex justify-end">
                <button onclick="closeTransactionModal()" 
                        class="bg-gray-300 hover:bg-gray-400 text-gray-700 px-4 py-2 rounded-md">
                    Close
                </button>
            </div>
        </div>
    </div>
</div>

<script>
function showTransactionDetails(transactionId) {
    document.getElementById('transactionModal').classList.remove('hidden');
    document.getElementById('transactionDetails').innerHTML = '<div class="text-center">Loading...</div>';
    
    fetch(`/api/transactions/${transactionId}`)
        .then(response => response.json())
        .then(data => {
            displayTransactionDetails(data);
        })
        .catch(error => {
            document.getElementById('transactionDetails').innerHTML = 
                '<div class="text-red-600">Error loading transaction details.</div>';
        });
}

function closeTransactionModal() {
    document.getElementById('transactionModal').classList.add('hidden');
}

function displayTransactionDetails(transaction) {
    const details = document.getElementById('transactionDetails');
    
    const formatDate = (dateString) => {
        const date = new Date(dateString);
        return date.toLocaleDateString() + ' at ' + date.toLocaleTimeString();
    };
    
    const getStatusBadge = (status) => {
        const classes = {
            'check_out': 'bg-yellow-100 text-yellow-800',
            'check_in': 'bg-green-100 text-green-800'
        };
        
        return `<span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${classes[status] || 'bg-gray-100 text-gray-800'}">
                    ${status === 'check_out' ? 'Check Out' : 'Check In'}
                </span>`;
    };
    
    const getConditionBadge = (condition) => {
        if (!condition) return '';
        
        const classes = {
            'excellent': 'bg-green-100 text-green-800',
            'good': 'bg-blue-100 text-blue-800',
            'fair': 'bg-yellow-100 text-yellow-800',
            'poor': 'bg-red-100 text-red-800',
            'damaged': 'bg-red-100 text-red-800'
        };
        
        return `<span class="px-2 inline-flex text-xs leading-4 font-medium rounded-full ${classes[condition] || 'bg-gray-100 text-gray-800'}">
                    ${condition.charAt(0).toUpperCase() + condition.slice(1)}
                </span>`;
    };
    
    details.innerHTML = `
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
                <h4 class="font-semibold text-gray-900 mb-2">Transaction Information</h4>
                <div class="space-y-2">
                    <div>
                        <span class="text-sm font-medium text-gray-500">Type:</span>
                        ${getStatusBadge(transaction.type)}
                    </div>
                    <div>
                        <span class="text-sm font-medium text-gray-500">Date & Time:</span>
                        <span class="text-sm text-gray-900">${formatDate(transaction.transaction_time)}</span>
                    </div>
                    <div>
                        <span class="text-sm font-medium text-gray-500">Staff Member:</span>
                        <span class="text-sm text-gray-900">${transaction.user.name}</span>
                    </div>
                </div>
            </div>
            
            <div>
                <h4 class="font-semibold text-gray-900 mb-2">Laptop Information</h4>
                <div class="space-y-2">
                    <div>
                        <span class="text-sm font-medium text-gray-500">Asset Tag:</span>
                        <span class="text-sm text-gray-900">${transaction.laptop.asset_tag}</span>
                    </div>
                    <div>
                        <span class="text-sm font-medium text-gray-500">Brand/Model:</span>
                        <span class="text-sm text-gray-900">${transaction.laptop.brand} ${transaction.laptop.model}</span>
                    </div>
                    <div>
                        <span class="text-sm font-medium text-gray-500">Barcode:</span>
                        <span class="text-sm text-gray-900 font-mono">${transaction.laptop.barcode}</span>
                    </div>
                </div>
            </div>
            
            <div>
                <h4 class="font-semibold text-gray-900 mb-2">Student Information</h4>
                <div class="space-y-2">
                    <div>
                        <span class="text-sm font-medium text-gray-500">Name:</span>
                        <span class="text-sm text-gray-900">${transaction.student.full_name}</span>
                    </div>
                    <div>
                        <span class="text-sm font-medium text-gray-500">Student ID:</span>
                        <span class="text-sm text-gray-900">${transaction.student.student_id}</span>
                    </div>
                    <div>
                        <span class="text-sm font-medium text-gray-500">Class:</span>
                        <span class="text-sm text-gray-900">${transaction.student.class ? transaction.student.class.name : 'No Class'}</span>
                    </div>
                </div>
            </div>
            
            <div>
                <h4 class="font-semibold text-gray-900 mb-2">Condition & Notes</h4>
                <div class="space-y-2">
                    ${transaction.condition_before ? `
                    <div>
                        <span class="text-sm font-medium text-gray-500">Condition Before:</span>
                        ${getConditionBadge(transaction.condition_before)}
                    </div>
                    ` : ''}
                    ${transaction.condition_after ? `
                    <div>
                        <span class="text-sm font-medium text-gray-500">Condition After:</span>
                        ${getConditionBadge(transaction.condition_after)}
                    </div>
                    ` : ''}
                    ${transaction.damage_description ? `
                    <div>
                        <span class="text-sm font-medium text-gray-500">Damage Description:</span>
                        <div class="text-sm text-red-600 bg-red-50 p-2 rounded mt-1">
                            ${transaction.damage_description}
                        </div>
                    </div>
                    ` : ''}
                    ${transaction.notes ? `
                    <div>
                        <span class="text-sm font-medium text-gray-500">Notes:</span>
                        <div class="text-sm text-gray-900 bg-gray-50 p-2 rounded mt-1">
                            ${transaction.notes}
                        </div>
                    </div>
                    ` : ''}
                </div>
            </div>
        </div>
    `;
}

// Close modal when clicking outside
document.addEventListener('click', function(e) {
    const modal = document.getElementById('transactionModal');
    if (e.target === modal) {
        closeTransactionModal();
    }
});

// Close modal on Escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeTransactionModal();
    }
});

// Auto-set date filters to show last 30 days if no filters are set
document.addEventListener('DOMContentLoaded', function() {
    const urlParams = new URLSearchParams(window.location.search);
    const hasFilters = urlParams.has('date_from') || urlParams.has('date_to') || 
                      urlParams.has('type') || urlParams.has('student_search');
    
    if (!hasFilters) {
        const today = new Date();
        const thirtyDaysAgo = new Date(today.getTime() - (30 * 24 * 60 * 60 * 1000));
        
        document.querySelector('input[name="date_from"]').value = thirtyDaysAgo.toISOString().split('T')[0];
        document.querySelector('input[name="date_to"]').value = today.toISOString().split('T')[0];
    }
});
</script>
@endsection